from renki.core.lib.history_meta import Versioned
from sqlalchemy.orm import relationship
from sqlalchemy import Column, Integer, Boolean, ForeignKey
from sqlalchemy.orm.exc import NoResultFound
from renki.core.lib.database.table import RenkiDataTable
from sqlalchemy.ext.declarative import declared_attr
from renki.core.lib.communication.ticket_tables import TicketGroup
from renki.core.lib.communication.ticket import create_ticket_member, create_delete_ticket_member
from renki.core.lib.exceptions import Invalid, SoftLimitReached, HardLimitReached
from renki.core.lib.auth.db import DefaultLimits #Limits, DefaultLimits


class RenkiMemberDataTable(RenkiDataTable, Versioned):
    __abstract__ = True

    @declared_attr
    def member_id(cls):
        return Column('member_id', Integer, ForeignKey("members.id"), nullable=False)

    waiting = Column('waiting', Boolean, nullable=False, default=False)

    soft_limit = None
    hard_limit = None

    @declared_attr
    def ticket_group_id(cls):
        return Column(Integer, ForeignKey('ticket_groups.id'))

    @declared_attr
    def ticket_group(cls):
        return relationship(TicketGroup)

    def real_save(self, commit=False):
        return RenkiDataTable.save(self, commit)

    def save(self, commit=False):
        if not self.waiting:
            create_ticket_member(self)
        else:
            return RenkiDataTable.save(self, commit)

    def delete(self):
        """
        Delete this object from database
        """
        create_delete_ticket_member(self)
        return RenkiDataTable.delete(self)

    # Needs to be overwritten for tables that don't have service_id
    def get_service_id(self):
        if 'service_id' in self.__table__.columns:
            return self.service_id

        raise Invalid("No service_id")

    @classmethod
    def get_limits_for_user(cls, user_id):
        limits = {'soft_limit': cls.soft_limit, 'hard_limit': cls.hard_limit}

        """try:
            default_limit = DefaultLimits.query().filter(DefaultLimits.table == cls.__tablename__).one()
            limits = {'soft_limit': default_limit.soft_limit, 'hard_limit': default_limit.hard_limit}
        except NoResultFound:
            pass

        try:
            limit = Limits.query().filter(Limits.table == cls.__tablename__ and Limits.user_id == user_id).one()
            limits = {'soft_limit': limit.soft_limit, 'hard_limit': limit.hard_limit}
        except NoResultFound:
            pass"""

        return limits

    @classmethod
    def count_user_entries(cls, user_id):
        return cls.query.filter(cls.user_id == user_id).count()

    @classmethod
    def validate_add(cls, user, user_id):
        if user.has_permission('pass_limits'):
            return

        limits = cls.get_limits_for_user(user_id)
        entries = cls.count_user_entries(user_id)
        if entries >= limits['hard_limit']:
            raise HardLimitReached("Hard limit for ports reached")
        elif entries >= limits['soft_limit']:
            raise SoftLimitReached("Soft limit for ports reached")
