from flask.ext.script import Manager, Option, Command
from flask.ext.script.commands import InvalidCommand

from renki.core.lib.commands.command_utils import *
from renki.core.lib.database.table import db

import logging

logger = logging.getLogger("service_command")

service_manager = Manager(usage="Manage services")


class ListServices(Command):
    """
    List services in database
    """

    def run(self):
        print("{:<40} {:<25} {:<50}".format("name", "type", "servers"))
        print("-"*80)
        for service in sorted(get_services(), key=lambda x: x.name):
            print("{:<40} {:<25} {:<50}".format(service.name, service.service_type,
                                                ', '.join([x.name for x in service.servers])))


class AddService(Command):
    """
    Add service
    """

    option_list = (
        Option('type', help="Service type"),
        Option('name', help="Service name"),
    )

    def run(self, name, type):
        service_obj = get_service(name)

        if service_obj:
            raise InvalidCommand("Service already created")

        service_obj = get_or_create_service(name, type, [])

        service_obj.save()

        db.session.commit()

        logger.info("Service %s added using commandline" % name)


class RemoveService(Command):
    """
    Remove service
    """

    option_list = (
        Option('service', help="Service name"),
    )

    def run(self, service):
        service_obj = get_service(service)
        if not service_obj:
            raise InvalidCommand("Service %s not found" % service)

        db.session.delete(service_obj)
        db.session.commit()

        logger.info("Service %s removed using commandline" % service)


class ServiceAddServer(Command):
    """
    Add server to service
    """

    option_list = (
        Option('server', help="Server to add"),
        Option('service')
    )

    def run(self, server, service):
        service_obj = get_service(service)
        if not service_obj:
            raise InvalidCommand("Service %s not found" % service)

        server_obj = get_server(server)

        if not server_obj:
            raise InvalidCommand("Server %s not found" % server)

        if server_obj in service_obj.servers:
            raise InvalidCommand("Server %s already added" % server)

        service_obj.servers.append(server_obj)
        service_obj.save()

        db.session.commit()


class ServiceRemoveServer(Command):
    """
    Add server to service
    """

    option_list = (
        Option('server', help="Server to remove"),
        Option('service')
    )

    def run(self, server, service):
        service_obj = get_service(service)
        if not service_obj:
            raise InvalidCommand("Service %s not found" % service)

        server_obj = get_server(server)

        if not server_obj:
            raise InvalidCommand("Server %s not found" % server)

        if server_obj not in service_obj.servers:
            raise InvalidCommand("Server %s not in service" % server)

        service_obj.servers.remove(server_obj)
        service_obj.save()

        db.session.commit()


service_manager.add_command("list", ListServices)
service_manager.add_command("add", AddService)
service_manager.add_command("remove", RemoveService)
service_manager.add_command("add-server", ServiceAddServer)
service_manager.add_command("remove-server", ServiceRemoveServer)
