from renki.core.lib.utils import ok, error, unauthorized, not_allowed, not_found, conflict, forbidden, server_error
from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.database.table import db
from flask import Blueprint
import logging

logger = logging.getLogger('default_routes')

blueprint = Blueprint('default', __name__)


@renkiapi('GET', '/',
          response={'info': {'type': 'string'}},
          require_authentication=False, blueprint=blueprint)
def index_route():
    """
    Front page for Renki.
    """
    return ok({'info': 'Renki service management system API'})


@renkiapi('GET', '/version',
          response={'version': {'type': 'string'}},
          require_authentication=False, blueprint=blueprint)
def version_route():
    """
    Show version
    """
    return ok({'version': '0.0.1'})


@renkiapi(['GET', 'POST', 'PUT', 'DELETE'], "/error", response={}, require_authentication=False, blueprint=blueprint)
def error_route():
    """
    Dummy route to represent error response
    """
    return error('This route fails always')


def get_error_str(data):
    try:
        if data.body:
            data = data.body
    except AttributeError:
        try:
            data = data.msg
        except AttributeError:
            pass
    return str(data)


def error400(data):
    db.session.rollback()
    return error('Request is invalid', data={'info': get_error_str(data)})


def error401(data):
    db.session.rollback()
    return unauthorized('Authentication required', data={'info': get_error_str(data)})


def error403(data):
    db.session.rollback()
    return forbidden('Permission denied', data={'info': get_error_str(data)})


def error404(data):
    db.session.rollback()
    return not_found('Requested page not found', data={'info': get_error_str(data)})


def error405(data):
    db.session.rollback()
    return not_allowed('Method not allowed', data={'info': get_error_str(data)})


def error409(data):
    db.session.rollback()
    return conflict('Conflict', data={'info': get_error_str(data)})


def error500(data):
    db.session.rollback()
    return server_error('Internal server error occurred', data={'info': get_error_str(data)})


def setup_error_handlers(app):
    app.register_error_handler(400, error400)
    app.register_error_handler(401, error401)
    app.register_error_handler(403, error403)
    app.register_error_handler(404, error404)
    app.register_error_handler(405, error405)
    app.register_error_handler(409, error409)
    app.register_error_handler(500, error500)
