from renki.core.lib.renkiapi import renkiapi
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.auth.db import UserMemberPermissionGroup, User, Member, UserToMember, Permission
from renki.core.lib.utils import ok as ret_ok, conflict as ret_conflict, forbidden as ret_denied,\
    not_found as ret_not_found
from renki.core.lib.auth.basic_permissions import ViewUserMemberPermissionGroups, EditUserMemberPermissionGroups
from sqlalchemy.orm.exc import NoResultFound
from renki.core.lib.database.table import db
from flask import Blueprint

blueprint = Blueprint('user_member_permission_group', __name__)


@renkiapi('GET', '/permission_groups/user_member',
          response={
              'type': 'object',
              'properties': {
                  'user_permission_groups': {
                      'type': 'array',
                      'items': {
                          'type': 'object'
                      }
                  }
              },
              'required': ['user_permission_groups'],
              'additionalProperties': False
          }, description='List user-member permission groups', blueprint=blueprint)
def permission_groups_user_member_list(identity):
    """
    List all user member permission groups
    :returns list of user member permission groups in following format:
    user_member_permission_groups: {
        1: {
            name: group1,
            description: Group one is a group,
        }
    }
    """
    if not identity.has_permission(ViewUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    user_member_permission_groups = {}

    for user_member_permission_group in UserMemberPermissionGroup.query.all():
        user_member_permission_groups[user_member_permission_group.id] = {
            'name': user_member_permission_group.name,
            'description': user_member_permission_group.description
        }

    return ret_ok({'user_member_permission_groups': user_member_permission_groups})


@renkiapi('POST', '/permission_groups/user_member',
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'description': {'type': 'string'}
              },
              'required': ['name', 'description'],
              'additionalProperties': False
          },
          response={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_add(identity, data):
    """
    Create new user-member permission group with name 'name' and description 'description'
    """
    if not identity.has_permission(EditUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    name = data['name']
    description = data['description']

    if UserMemberPermissionGroup.query.filter(UserMemberPermissionGroup.name == name).count():
        return ret_conflict('User-member permission group with name %s already exists' % name)

    permission_group = UserMemberPermissionGroup()
    permission_group.name = name
    permission_group.description = description
    permission_group.save()

    db.session.commit()

    return ret_ok({'permission_group_id': permission_group.id})


@renkiapi('DELETE', '/permission_groups/user_member/<int:permission_group_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_delete(identity, data):
    """
    Delete user-member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    db.session.delete(permission_group)
    db.session.commit()

    return ret_ok()


@renkiapi('PUT', '/permission_groups/user_member/<int:permission_group_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'name': {'type': 'string'},
                  'description': {'type': 'string'}
              },
              'required': ['name', 'description'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_update(identity, data):
    """
    Update name and description of user-member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    name = data['name']
    description = data['description']

    permission_group.name = name
    permission_group.description = description
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('GET', '/permission_groups/user_member/<int:permission_group_id>/user_members',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_users_list(identity, data):
    """
    List users in user-member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(ViewUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    user_members = []

    for user_member in permission_group.user_members:
        user_members.append([user_member.user_id, user_member.member_id])

    return ret_ok({'user_members': user_members})


@renkiapi('POST', '/permission_groups/user_member/<int:permission_group_id>/user_members',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'user_id': {'type': 'integer'},
                  'member_id': {'type': 'integer'}
              },
              'required': ['user_id', 'member_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_user_members_add(identity, data):
    """
    Add user-member connection with user_id 'user_id' and member_id 'member_id'
    to user-member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    user_id = data['user_id']
    try:
        user = User.get(user_id)
    except DoesNotExist:
        return ret_not_found('User with id %i not found' % user_id)

    member_id = data['member_id']
    try:
        member = Member.get(member_id)
    except DoesNotExist:
        return ret_not_found('Member with id %i not found' % member_id)

    try:
        user_to_member = UserToMember.query.filter(
            UserToMember.user_id == user.id, UserToMember.member_id == member.id).one()
    except NoResultFound:
        user_to_member = UserToMember()
        user_to_member.user_id = user.id
        user_to_member.member_id = member.id

    permission_group.user_members.append(user_to_member)
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('DELETE',
          '/permission_groups/user_member/<int:permission_group_id>/user_members/<int:user_id>/<int:member_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'},
                  'user_id': {'type': 'integer'},
                  'member_id': {'type': 'integer'}
              },
              'required': ['permission_group_id', 'user_id', 'member_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_users_delete(identity, data):
    """
    Delete user-member connection with user_id 'user_id' and member_id 'member_id'
    from user-member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    user_id = int(data['user_id'])
    member_id = int(data['member_id'])

    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    try:
        user = User.get(user_id)
    except DoesNotExist:
        return ret_not_found('User with id %i not found' % user_id)

    try:
        member = Member.get(member_id)
    except DoesNotExist:
        return ret_not_found('Member with id %i not found' % member_id)

    try:
        user_to_member = UserToMember.query.filter(
            UserToMember.user_id == user.id, UserToMember.member_id == member.id).one()
    except NoResultFound:
        return ret_not_found('User to member connection not found')

    permission_group.user_members.remove(user_to_member)
    db.session.commit()

    return ret_ok()


@renkiapi('GET', '/permission_groups/user_member/<int:permission_group_id>/permissions',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_permissions_list(identity, data):
    """
    List permissions in user-member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(ViewUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    permissions = {}

    for permission in permission_group.permissions:
        permissions[permission.id] = {
            'name': permission.name,
            'description': permission.description,
            'service_id': permission.service_id,
            'is_global': permission.is_global
        }

    return ret_ok({'permissions': permissions})


@renkiapi('POST', '/permission_groups/user_member/<int:permission_group_id>/permissions',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'}
              },
              'required': ['permission_group_id'],
              'additionalProperties': False
          },
          json={
              'type': 'object',
              'properties': {
                  'permission_id': {'type': 'integer'}
              },
              'required': ['permission_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_permissions_add(identity, data):
    """
    Add permission with id permission_id to user-member permission group with id permission_group_id
    """
    if not identity.has_permission(EditUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])

    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    permission_id = data['permission_id']

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    permission_group.permissions.append(permission)
    permission_group.save()
    db.session.commit()

    return ret_ok()


@renkiapi('DELETE', '/permission_groups/user_member/<int:permission_group_id>/permissions/<int:permission_id>',
          url_params={
              'type': 'object',
              'properties': {
                  'permission_group_id': {'type': 'integer'},
                  'permission_id': {'type': 'integer'}
              },
              'required': ['permission_group_id', 'permission_id'],
              'additionalProperties': False
          }, blueprint=blueprint)
def permission_groups_user_member_permissions_delete(identity, data):
    """
    Delete permission with id 'permission_id' from user-member permission group with id 'permission_group_id'
    """
    if not identity.has_permission(EditUserMemberPermissionGroups):
        return ret_denied('Not allowed')

    permission_group_id = int(data['permission_group_id'])
    permission_id = int(data['permission_id'])

    try:
        permission_group = UserMemberPermissionGroup.get(permission_group_id)
    except DoesNotExist:
        return ret_not_found('User-member permission group with id %i not found' % permission_group_id)

    try:
        permission = Permission.get(permission_id)
    except DoesNotExist:
        return ret_not_found('Permission with id %i not found' % permission_id)

    permission_group.permissions.remove(permission)
    permission_group.save()
    db.session.commit()

    return ret_ok()
