from renki.core.lib.utils import ok, error
from renki.core.lib import exceptions
from renki.core.lib.auth.db import AuthTokens
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.utils import unauthorized
from flask import request, Blueprint
import jsonschema
import inspect
import logging

logger = logging.getLogger('renkiapi')

__all__ = ['route_specifications', 'renkiapi', 'reset_route_specification']

spec_blueprint = Blueprint('spec', __name__)
route_specifications = {}


def add_route_specification(path, method, specification):
    global route_specifications, spec_blueprint
    if path not in route_specifications:
        route_specifications[path] = {
            'methods': {
                'GET': None,
                'POST': None,
                'PUT': None,
                'DELETE': None,
            },
            'url': path
        }

        def spec_route(*args, **kwargs):
            _ = args  # Suppress warning
            _ = kwargs  # Suppress warning
            return ok(route_specifications[path])

        spec_blueprint.route('/spec/%s' % path.lstrip('/'), methods=['GET'],
                             endpoint='/spec/%s' % path.lstrip('/'))(spec_route)
    route_specifications[path]['methods'][method] = specification


def reset_route_specification():
    global route_specifications, spec_blueprint
    for i in list(route_specifications.keys()):
        del route_specifications[i]

    spec_blueprint = Blueprint('spec', __name__)


def renkiapi(methods, *paths, url_params=None, json=None, response=None, description=None, app=None,
             require_authentication=True, blueprint=None):
    """
    This function is used as an decorator to define Renki routes.
    Input function is wrapped to do variable and permission validation on
    every call.

    :param methods: list of allowed HTTP methods
    :param paths: one or more path(s)
    :param url_params: parameters passed as part of the url
    :param json: POST/JSON parameters
    :param response: return value parameters
    :param description: Optional description. If None given, parsed from method docstring.
    :param app: Optional Flask application
    :param require_authentication: Set to false if non-authenticated users should be able to access
    :param blueprint: Which blueprint the URL belongs to

    Usage:

    from renki.core.lib import api

    @api.renkiapi("GET", "/foo/<foo:str>/<bar:int>",
                  get={
                      'foo': {'type': api.String},
                      'bar': {'type': api.Integer},
                  },
                  response={
                      'foo': {'type': api.String},
                  })
    def my_route(foo, bar):
        return {'foo': foo}

    @api.renkiapi("POST", "/foo/",
                  json={
                       'foo': {'type': api.String},
                       'bar': {'type': api.Integer},
                  },
                  data_variable = 'data',
                  response={
                      'foo': {'type': api.String},
                  })
    def my_route(data):
        return {'foo': data['foo']}
    """
    route_methods = []
    if isinstance(methods, list):
        for method in methods:
            if method not in ['GET', 'POST', 'PUT', 'DELETE']:
                raise exceptions.RenkiBug('Invalid method given for apispec decorator')
            route_methods.append(method)
    elif isinstance(methods, str):
        if methods not in ['GET', 'POST', 'PUT', 'DELETE']:
            raise exceptions.RenkiBug('Invalid method given for apispec decorator')
        route_methods.append(methods)
    else:
        raise exceptions.RenkiBug('Method should be either string or list of strings')

    for x in paths:
        if not isinstance(x, str) or not x.startswith('/'):
            raise exceptions.RenkiBug('Invalid path %s' % x)

    paths = list(paths)
    if len(paths) == 0:
        raise exceptions.RenkiBug('No path given for apispec decorator')

    if isinstance(url_params, dict):
        try:
            jsonschema.Draft4Validator.check_schema(url_params)
        except jsonschema.exceptions.SchemaError:
            raise exceptions.RenkiBug('Invalid get-schema')
    elif url_params is not None:
        raise exceptions.RenkiBug('Invalid get argument, dict expected')

    if isinstance(json, dict):
        try:
            jsonschema.Draft4Validator.check_schema(json)
        except jsonschema.exceptions.SchemaError:
            raise exceptions.RenkiBug('Invalid json-schema')
    elif json is not None:
        raise exceptions.RenkiBug('Invalid json argument, dict expected')

    if isinstance(response, dict):
        try:
            jsonschema.Draft4Validator.check_schema(response)
        except jsonschema.exceptions.SchemaError:
            raise exceptions.RenkiBug('Invalid response-schema')
    elif response is not None:
        raise exceptions.RenkiBug('Invalid response argument, dict expected')

    def decorator(f):
        """
        This wrapper is used to handle decorators inner function call. As following

        @renkiapi(*args, **kwargs)
        def my_call(foo):

        expands to "renkiapi(*args, **kwargs)(my_call)", this function handles "(my_call)" part.

        :param f: Input function
        :return: Input function with type and permission checking.
        """
        api_description = description
        f_spec = inspect.getfullargspec(f)

        if not api_description:
            api_description = f.__doc__
        if api_description:
            api_description = api_description.strip()
        else:
            api_description = 'No description available'

        spec = {
            'description': api_description,
            'response': response,
            'urlParams': url_params,
            'json': json
        }

        # Insert route specification for all given paths
        for path in paths:
            for route_method in route_methods:
                add_route_specification(path, route_method, spec)

        def function_wrapper(*args, **kwargs):
            # Determine identity if possible
            identity = None
            if 'Authorization' in request.headers:
                auth_token = request.headers['Authorization']
                try:
                    identity = AuthTokens.get_token(auth_token)
                except DoesNotExist:
                    if require_authentication:
                        logger.warning('Authentication failure from %s. Attempted to use user %s.'
                                       % (request.remote_addr, identity.user_id))
                        return unauthorized('Not authenticated')
            elif require_authentication:
                logger.warning('Authentication failure from %s. No authorization header present.' % request.remote_addr)
                return unauthorized('Not authenticated')

            get_data = {}

            # Move values that are passed inside URL from kwargs to data dictionary
            for key, value in kwargs.items():
                get_data[key] = value

            kwargs = {}

            for key in request.args:
                get_data[key] = request.args.get(key)

            try:
                if url_params is not None:
                    jsonschema.validate(get_data, url_params)

                data = get_data.copy()

                post_data = {}
                if request.json is not None:
                    post_data.update(request.json)
                else:
                    for key in request.form:
                        post_data[key] = request.form.get(key)

                if json is not None:
                    jsonschema.validate(post_data, json)
            except jsonschema.exceptions.ValidationError as e:
                logger.exception('Schema validation failed')
                return error(str(e))

            data.update(post_data)

            # Check if route actually requires identity or data to avoid errors
            # This is mainly for avoiding warnings about unused parameters
            if 'identity' in f_spec.args:
                kwargs['identity'] = identity

            if 'data' in f_spec.args:
                kwargs['data'] = data

            return_val = f(*args, **kwargs)
            return return_val

        function_wrapper.__name__ = f.__name__
        function_wrapper.__doc__ = f.__doc__

        for path in paths:
            if app is not None:
                app.add_url_rule(path, view_func=function_wrapper, methods=route_methods)

            if blueprint is not None:
                blueprint.add_url_rule(path, view_func=function_wrapper, methods=route_methods)

        return function_wrapper

    return decorator
