import logging
import crowd

from datetime import datetime
from renki.core import settings
from renki.core.lib.auth.authentication import AuthenticationModule
from renki.core.lib.exceptions import AuthenticationFailed

logger = logging.getLogger('CrowdAuthentication')


class CrowdAuthenticationModule(AuthenticationModule):
    """
    CrowdAuthenticationModule provides authentication against an Atlassian Crowd instance
    """
    NAME = "CROWD"

    @staticmethod
    def get_crowd():
        return crowd.CrowdServer(
            settings.CROWD_URL,
            settings.CROWD_USER,
            settings.CROWD_PASS
        )

    @classmethod
    def _register_user(cls, username: str, password: str, params: dict):
        return False

    @classmethod
    def _authenticate(cls, username, password):
        """
        Authenticate user using username and password
        :return: token associated with session
        """
        session = cls.get_crowd().get_session(username, password)
        if session is None:
            raise AuthenticationFailed("Invalid username or password")

        # Parse expiration datetime
        expiry_timestamp = session['expiry-date']
        expiry_date = datetime.utcfromtimestamp(expiry_timestamp / 1e3)

        return {'token': session['token'], 'expires': expiry_date}

    @classmethod
    def _validate_token(cls, token):
        """
        Checks if token is valid.
        For basic module this is always true as token has been generated and put straight to AuthTokens
        """
        success = cls.get_crowd().validate_session(token)
        if success is not None:
            return True
        else:
            return False

    @classmethod
    def require_password(cls):
        return False
