from renki.core.lib.database.table import RenkiTable
from renki.core.lib.database.tables import register_table
from renki.core.lib.database.basic_tables import Service
from renki.core.lib import renki_settings as settings
from renki.core.lib.exceptions import DoesNotExist
from renki.core.lib.utils import generate_token
from renki.core.lib.auth import permissions
from renki.core.lib.database.table import db
from sqlalchemy.orm import relationship, backref
from sqlalchemy.orm.exc import NoResultFound, MultipleResultsFound
from sqlalchemy.exc import SQLAlchemyError
from sqlalchemy import Column, Unicode, Integer, DateTime, ForeignKey, Boolean, or_, UniqueConstraint
from datetime import datetime, timedelta
from hashlib import sha512
import logging

logger = logging.getLogger("auth.db")


def expires_funct():
    return datetime.now() + timedelta(seconds=settings.KEY_EXPIRE_TIME)


def hash_password(password, salt=None):
    if isinstance(password, str):
        password = password.encode("utf-8")
    if salt is None:
        salt = generate_token(size=20)
    return "$1$%s$%s" % (salt, sha512(salt.encode("utf-8") + password).hexdigest())


class Member(RenkiTable):
    __tablename__ = 'members'
    id = Column('id', Integer, nullable=False, primary_key=True)

    def validate(self):
        return True

register_table(Member)


class BasicAuthUser(RenkiTable):
    __tablename__ = 'basic_auth_users'
    id = Column('id', Integer, nullable=False, primary_key=True)
    name = Column('name', Unicode, nullable=False)
    password = Column('password', Unicode, nullable=False)

    def validate(self):
        return True

    def set_password(self, password):
        """
        Change password of user in local database
        :param password: new password for user
        """
        self.password = hash_password(password)

    def check_password(self, password):
        """
        Check if password is valid
        :param password:
        :return: true if password is valid
        """
        try:
            _, pwtype, salt, pwhash = self.password.split('$', 3)
        except Exception as e:
            logger.exception(e)
            return False
        return self.password == hash_password(password, salt=salt)

register_table(BasicAuthUser)


class User(RenkiTable):
    """
    Users database table
    """
    __tablename__ = 'users'
    added = Column('added', DateTime, default=datetime.now)
    name = Column('name', Unicode, nullable=False)
    auth_module = Column('auth_module', Unicode, nullable=False)
    __table_args__ = (UniqueConstraint('name', 'auth_module'),)

    @property
    def user_id(self):
        return self.id

    def validate(self):
        return True

    def get_member(self):
        """
        Get member that user can act as or raise an exception if user can act as multiple members
        """
        try:
            return UserToMember.query().filter(UserToMember.user_id == self.id).one()
        except (NoResultFound, MultipleResultsFound):
            raise

    def get_members(self):
        try:
            return UserToMember.query().filter(UserToMember.user_id == self.id).all()
        except NoResultFound:
            raise

register_table(User)


class AuthTokens(RenkiTable):
    """
    Database table for authentication token-user pairing
    """
    __tablename__ = 'auth_tokens'
    token = Column('token', Unicode, nullable=False)
    expires = Column('expires', DateTime, nullable=False, default=expires_funct)
    user_id = Column('user_id', Integer, ForeignKey('users.id'), nullable=False)
    user = relationship('User', backref='auth_tokens')
    member_id = Column('member_id', Integer, ForeignKey('members.id'), nullable=True)
    member = relationship('Member', backref='auth_tokens')

    def validate(self):
        return True

    @classmethod
    def get_user_by_token(cls, token):
        try:
            return cls.query().filter(AuthTokens.token == token).one().user
        except NoResultFound:
            return None

    @classmethod
    def add_token(cls, user, token, expires=None):
        a = AuthTokens()
        a.token = token
        a.user_id = user.id
        if expires is not None:
            a.expires = expires
        db.session.add(a)
        return a

    @classmethod
    def delete_token(cls, token):
        try:
            item = cls.query().filter(AuthTokens.token == token).one()
        except SQLAlchemyError as e:
            logger.exception(e)
            raise DoesNotExist("Token does not exist")
        item.delete()

    @classmethod
    def get_token(cls, token):
        try:
            item = cls.query.filter(AuthTokens.token == token, or_(
                                      AuthTokens.expires > datetime.now(),
                                      AuthTokens.expires is None)).one()
        except SQLAlchemyError as e:
            logger.exception(e)
            raise DoesNotExist("Token does not exist")
        return item

    def get_permissions_query(self):
        user_permissions = db.session.query(Permission).select_from(User).filter(User.id == self.user_id).\
            join(user_to_user_permission_group).join(UserPermissionGroup).\
            join(user_permission_groups_to_permissions).join(Permission)

        if self.member_id is not None:
            user_member_permissions = db.session.query(Permission).select_from(UserToMember).\
                filter(UserToMember.user_id == self.user_id, UserToMember.member_id == self.member_id).\
                join(user_members_to_user_member_permission_groups).\
                join(UserMemberPermissionGroup).\
                join(user_member_permission_groups_to_permissions).join(Permission)

            member_permissions = db.session.query(Permission).select_from(Member).filter(Member.id == self.member_id).\
                join(member_to_member_permission_groups).\
                join(MemberPermissionGroup).\
                join(member_permission_groups_to_permissions).join(Permission)

            available_member_permissions = user_member_permissions.intersect(member_permissions)
            available_permissions = user_permissions.union(available_member_permissions)
        else:
            available_permissions = user_permissions

        return available_permissions

    def get_permissions_unique_to_service(self):
        return self.get_permissions_query().distinct(Permission.service_id).all()

    def get_permissions_with_unique_service_ids(self, name):
        return self.get_permissions_query().filter(Permission.name == name).distinct(Permission.service_id).all()

    def get_permissions(self):
        return self.get_permissions_query().all()

    def has_permission(self, permission, service_id=None):
        if isinstance(permission, permissions.GlobalPermission):
            if service_id is not None:
                logger.error('has_permission with GlobalPermission doesn\'t expect service_id')
                return False
            if self.get_permissions_query().filter(Permission.name == permission.name).count():
                return True
            return False
        elif isinstance(permission, permissions.ServicePermission):
            if service_id is None:
                if self.get_permissions_query().filter(Permission.name == permission.name).count() > 0:
                    return True

            if self.get_permissions_query().\
                    filter(Permission.name == permission.name, Permission.service_id == service_id).count():
                return True
            return False
        elif isinstance(permission, str):
            if service_id is not None:
                logger.warning('Assuming checking for service permission since service_id is not None')
                if self.get_permissions_query().\
                        filter(Permission.name == permission, Permission.service_id == service_id).count():
                    return True
                return False
            else:
                logger.warning('Assuming checking for global permission since service_id is None')
                if self.get_permissions_query().filter(Permission.name == permission).count():
                    return True
                return False

        logger.error('Trying to use has_permission with invalid permission object')

        return False

register_table(AuthTokens)

# Map user_permission_groups to permissions
user_permission_groups_to_permissions = db.Table(
    'user_permission_groups_to_permissions',
    Column('user_permission_group_id', Integer,  ForeignKey('user_permission_groups.id')),
    Column('permission_id', Integer, ForeignKey('permissions.id'))
)

# Map member permission groups to permissions
member_permission_groups_to_permissions = db.Table(
    'member_permission_groups_to_permissions',
    Column('member_permission_group_id', Integer, ForeignKey('member_permission_groups.id')),
    Column('permission_id', Integer, ForeignKey('permissions.id')))

# Map user_member_permission_groups to permissions
user_member_permission_groups_to_permissions = db.Table(
    'user_member_permission_groups_to_permissions',
    Column('user_member_permission_group_id', Integer, ForeignKey('user_member_permission_groups.id')),
    Column('permission_id', Integer, ForeignKey('permissions.id'))
)

# Map users to user_permission_groups
user_to_user_permission_group = db.Table(
    'user_to_user_permission_group',
    Column('user_id', Integer, ForeignKey('users.id')),
    Column('user_permission_group_id', Integer, ForeignKey('user_permission_groups.id'))
)

# Map users_to_members to user_member_permission_groups
user_members_to_user_member_permission_groups = db.Table(
    'user_member_to_user_member_permission_group',
    Column('users_to_member_id', Integer, ForeignKey('users_to_members.id')),
    Column('user_member_permission_group_id', Integer, ForeignKey('user_member_permission_groups.id'))
)

# Map members to member_permission_groups
member_to_member_permission_groups = db.Table(
    'member_to_member_permission_groups',
    Column('member_id', Integer, ForeignKey('members.id')),
    Column('member_permission_group_id', Integer, ForeignKey('member_permission_groups.id'))
)


class GroupTable(RenkiTable):
    __abstract__ = True

    def add_permission(self, permission):
        """
        Add permission

        :param permission: Permission object
        :return: None
        """
        if not isinstance(permission, Permission):
            raise ValueError("Invalid permission %s, should be Permission object" % (permission,))

        if permission in self.permissions:
            return

        logging.warning("Adding permission %s to group %s %s" % (permission.name, self.__class__.__name__, self.name))
        self.permissions.append(permission)

    def remove_permission(self, permission):
        """
        Remove permission

        :param permission: Permission object
        :return: None
        """
        if not isinstance(permission, Permission):
            raise ValueError("Invalid permission %s, should be Permission object" % (permission,))

        if permission not in self.permissions:
            logger.info("Cannot remove permission %s from group %s, permission not added" % (permission.name,
                                                                                             self.__class__.__name__))
            return

        logging.warning("Removing permission %s from group %s %s" % (permission.name, self.__class__.__name__,
                                                                     self.name))
        self.permissions.remove(permission)

    def add_member(self, member):
        raise NotImplementedError("Not Implemented")

    def remove_member(self, member):
        raise NotImplementedError("Not Implemented")


class UserToMember(RenkiTable):
    """
    Map users to members with relation to user_member_permission_group
    """
    __tablename__ = 'users_to_members'

    user_id = Column('user_id', Integer, ForeignKey('users.id'), nullable=False)
    member_id = Column('member_id', Integer, ForeignKey('members.id'), nullable=False)
    __table_args__ = (UniqueConstraint('user_id', 'member_id'),)

    @classmethod
    def add_user_to_member(cls, user, member):
        logging.info("Adding access to member %s for user %s" % (member.id, user.id))
        new = UserToMember()
        new.user_id = user.id
        new.member_id = member.id
        db.session.add(new)
        return new

    @classmethod
    def remove_user_from_user_to_member(cls, user, member):
        logging.info("Removing access to member %s from user %s" % (member.id, user.id))
        try:
            user_member = UserToMember().query.filter(UserToMember.user_id == user.id,
                                                      UserToMember.member_id == member.id).one()
            user_member.remove()
        except NoResultFound:
            pass
        return

    def validate(self):
        return True

    def __str__(self):
        return "UserToMember<%s,%s>" % (self.user_id, self.member_id)


register_table(UserToMember)


class Permission(RenkiTable):
    """
    This table list permissions.
    Permissions are generated from code by admin.py.
    Examples:
        - add_port
        - view_port
    """
    __tablename__ = 'permissions'
    name = Column('name', Unicode, nullable=False, unique=False)
    description = Column('description', Unicode, nullable=False, default="")
    service_id = Column('service_id', Integer, ForeignKey(Service.id), nullable=True)
    is_global = Column('is_global', Boolean, default=False)
    __table_args__ = (UniqueConstraint('name', 'service_id'),)

    def validate(self):
        return True

register_table(Permission)


class UserPermissionGroup(GroupTable):
    """
    Table for user permission groups
    Examples:
        - administrator
        - normal (user)
    """
    __tablename__ = 'user_permission_groups'
    name = Column('name', Unicode, nullable=False, unique=True)
    description = Column('description', Unicode, nullable=False, default="")
    permissions = relationship("Permission", secondary=user_permission_groups_to_permissions,
                               backref=backref("user_permission_groups", lazy="dynamic"))
    users = relationship("User", secondary=user_to_user_permission_group,
                         backref=backref("user_permission_groups", lazy="dynamic"))

    def add_user(self, user):
        """
        Add user to user permission group
        :param user: Users object
        :return: None
        """
        if not isinstance(user, User):
            raise ValueError("Invalid user %s, should be Users object" % (user,))

        if user in self.users:
            return

        logging.info("Adding user %s to UserPermissionGroup %s" % (user.id, self.name))
        self.users.append(user)

    def remove_user(self, user: User):
        """
        Remove user from user permission group
        :param user: User object
        :return: None
        """
        if not isinstance(user, User):
            raise ValueError("Invalid user %s, should be Users object" % (user,))

        if user not in self.users:
            return

        logger.info("Removing user %s from UserPermissionGroup %s" % (user.id, self.name))
        self.users.remove(user)

    add_member = add_user
    remove_member = remove_user

    def validate(self):
        return True

register_table(UserPermissionGroup)


class UserMemberPermissionGroup(GroupTable):
    """
    Table for user member permissions groups
    Examples:
        - technical
        - billing
    """
    __tablename__ = 'user_member_permission_groups'
    name = Column('name', Unicode, nullable=False, unique=True)
    description = Column('description', Unicode, nullable=False, default="")
    permissions = relationship("Permission", secondary=user_member_permission_groups_to_permissions,
                               backref=backref("user_member_permission_groups", lazy="dynamic"))
    user_members = relationship("UserToMember", secondary=user_members_to_user_member_permission_groups,
                                backref=backref("user_member_permission_groups", lazy="dynamic"))

    def add_user_member(self, user_member):
        if not isinstance(user_member, UserToMember):
            raise ValueError("Invalid user_member %s, should be UserToMember object" % (user_member,))

        if user_member in self.user_members:
            return

        logging.info("Adding user_member %s to UserMemberPermissionGroup %s" % (user_member.id, self.name))
        self.user_members.append(user_member)

    def remove_user_member(self, user_member):
        if not isinstance(user_member, UserToMember):
            raise ValueError("Invalid user_member %s, should be UserToMember object" % (user_member,))

        if user_member not in self.user_members:
            return

        logging.info("Removing user_member %s from UserMemberPermissionGroup %s" % (user_member.id, self.name))
        self.user_members.remove(user_member)

    add_member = add_user_member
    remove_member = remove_user_member

    def validate(self):
        return True

register_table(UserMemberPermissionGroup)


class MemberPermissionGroup(GroupTable):
    """
    Table for member permission groups.
    Examples:
        - person
        - community
        - supporting
    """
    __tablename__ = 'member_permission_groups'
    name = Column('name', Unicode, nullable=False, unique=True)
    description = Column('description', Unicode, nullable=False, default="")
    permissions = relationship("Permission", secondary=member_permission_groups_to_permissions,
                               backref=backref("member_permission_groups", lazy="dynamic"))
    members = relationship("Member", secondary=member_to_member_permission_groups,
                           backref=backref("member_permission_groups", lazy="dynamic"))

    def add_member(self, member):
        """
        Add user to user permission group
        :param member: Members object
        :return: None
        """
        if not isinstance(member, Member):
            raise ValueError("Invalid member %s, should be Member object" % (member,))

        # Check if member is already in group
        if member in self.members:
            return

        logging.info("Adding member %s to MemberPermissionGroup %s" % (member.id, self.name))
        self.members.append(member)

    def remove_member(self, member):
        """
        Remove user from user permission group
        :param member: Members object
        :return: None
        """
        if not isinstance(member, Member):
            raise ValueError("Invalid member %s, should be Member object" % (member,))

        # Check if member is already removed
        if member not in self.members:
            return

        logging.info("Removing member %s from MemberPermissionGroup %s" % (member.id, self.name))
        self.members.remove(member)

    def validate(self):
        return True

register_table(MemberPermissionGroup)


class DefaultLimits(RenkiTable):
    __tablename__ = 'default_limits'
    table = Column('table', Unicode, nullable=False)
    soft_limit = Column('soft_limit', Integer)
    hard_limit = Column('hard_limit', Integer)

    def validate(self):
        return True

register_table(DefaultLimits)


class Limits(RenkiTable):
    __tablename__ = 'limits'
    limit_type = Column('type', Unicode)
    table = Column('table', Unicode, nullable=False)
    soft_limit = Column('soft_limit', Integer)
    hard_limit = Column('hard_limit', Integer)

    __mapper_args__ = {
        'polymorphic_identity': 'limits',
        'polymorphic_on': limit_type
    }

    def validate(self):
        return True

register_table(Limits)


class MemberLimits(Limits):
    __tablename__ = 'member_limits'
    id = Column(Integer, ForeignKey('limits.id'), primary_key=True)
    member_id = Column('member_id', Integer, ForeignKey('members.id'))

    __mapper_args__ = {
        'polymorphic_identity': 'member_limits'
    }

    def validate(self):
        return True

register_table(MemberLimits)
