from renki.core.lib.test_utils import BasicTest
from renki.core.lib.auth.permissions import ServicePermission
from renki.core.lib.auth.authentication_modules.basic import BasicAuthenticationModule
from renki.core.lib.auth.db import Permission, UserPermissionGroup
from renki.core.lib.auth.basic_permissions import EditPermissions, ViewPermissions
from renki.core.lib.database.table import db
import unittest
import jsonschema

CreatePortPermission = ServicePermission('create_port', 'Create port')


class PermissionRouteTest(BasicTest):
    def setUp(self):
        super(PermissionRouteTest, self).setUp()

        self._lakka = self.get_or_create_server('Lakka')
        db.session.flush()

        self._lakka_port = self.get_or_create_service('Lakka_port', 'port', [self._lakka])
        db.session.commit()

        self._view_permissions = Permission()
        self._view_permissions.name = ViewPermissions.name
        self._view_permissions.is_global = True
        self._view_permissions.save()

        self._edit_permissions = Permission()
        self._edit_permissions.name = EditPermissions.name
        self._edit_permissions.is_global = True
        self._edit_permissions.save()

        self._create_lakka_port = Permission()
        self._create_lakka_port.name = CreatePortPermission.name
        self._create_lakka_port.service_id = self._lakka_port.id
        self._create_lakka_port.is_global = False
        self._create_lakka_port.save()
        db.session.commit()

        self._basic_user_1 = BasicAuthenticationModule.register_user('basic1', 'basic1')
        self._basic_user_2 = BasicAuthenticationModule.register_user('basic2', 'basic2')
        db.session.commit()

        self._admin_permission_group = UserPermissionGroup()
        self._admin_permission_group.name = 'Admin'
        self._admin_permission_group.description = 'Admins'
        self._admin_permission_group.add_permission(self._edit_permissions)
        self._admin_permission_group.add_permission(self._view_permissions)
        self._admin_permission_group.add_user(self._basic_user_1)
        db.session.commit()


class TestAddPermissionRoute(PermissionRouteTest):
    def setUp(self):
        super(TestAddPermissionRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.post('/permissions', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        """
        Test that user without EditPermissions-permission cannot create permission
        """
        self.auth('basic2', 'basic2')

        assert self.post('/permissions',
                         {
                             'name': 'test_perm',
                             'description': 'Permission for testing',
                             'isGlobal': 1,
                             'serviceId': 0
                         }, expect_errors=True).status_int == 403

    def test_add_valid(self):
        """
        Test that added permission gets to the database
        """
        self.auth('basic1', 'basic1')

        data = self.post('/permissions',
                         {
                             'name': 'test_perm',
                             'description': 'Permission for testing',
                             'isGlobal': 1,
                             'serviceId': 0
                         }).json

        permission_id = data['permission']

        perm = Permission.query.filter(Permission.id == permission_id).one()

        assert perm.name == 'test_perm'
        assert perm.description == 'Permission for testing'
        assert perm.is_global
        assert perm.service_id is None

    def test_add_duplicate(self):
        """
        Test that adding same permission twice gives correct error
        """
        self.auth('basic1', 'basic1')

        data = self.post('/permissions',
                         {
                             'name': 'test_perm',
                             'description': 'Permission for testing',
                             'isGlobal': 1,
                             'serviceId': 0
                         }).json

        permission_id = data['permission']

        perm = Permission.query.filter(Permission.id == permission_id).one()

        assert perm.name == 'test_perm'
        assert perm.description == 'Permission for testing'
        assert perm.is_global
        assert perm.service_id is None

        assert self.post('/permissions',
                         {
                             'name': 'test_perm',
                             'description': 'Permission for testing',
                             'isGlobal': 1,
                             'serviceId': 0
                         }, expect_errors=True).status_int == 409


class TestListPermissionsRoute(PermissionRouteTest):
    _permissions_schema = {
        'permissions': {
            'type': 'object',
            'patternProperties': {
                '^[0-9]+$': {
                    'type': 'object',
                    'properties': {
                        'name': {
                            'type': 'string'
                        },
                        'description': {
                            'type': 'string'
                        },
                        'serviceId': {
                            'type': 'integer'
                        },
                        'isGlobal': {
                            'type': 'boolean'
                        }
                    },
                    'required': ['name', 'description', 'serviceId', 'isGlobal'],
                    'additionalProperties': False
                }
            }
        },
        'additionalProperties': True
    }

    def setUp(self):
        super(TestListPermissionsRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.get('/permissions', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        """
        Test that user without ViewPermissions-permission cannot update permission
        """
        self.auth('basic2', 'basic2')

        assert self.get('/permissions', {}, expect_errors=True).status_int == 403

    def test_valid_response(self):
        """
        Test that listing response is in expected format
        """
        self.auth('basic1', 'basic1')

        data = self.get('/permissions', {}).json

        jsonschema.validate(data, self._permissions_schema)

        permissions = data['permissions']
        assert '1' in permissions and '2' in permissions and '3' in permissions
        assert '123' not in permissions

        assert permissions['1']['name'] == ViewPermissions.name


class TestDeletePermissionRoute(PermissionRouteTest):
    def setUp(self):
        super(TestDeletePermissionRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.delete('/permissions/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        """
        Test that user without EditPermissions-permission cannot delete permission
        """
        self.auth('basic2', 'basic2')

        assert self.delete('/permissions/1', {}, expect_errors=True).status_int == 403

    def test_non_existent_permission(self):
        """
        Test that trying to delete non-existent permission gives correct error
        """
        self.auth('basic1', 'basic1')

        assert self.delete('/permissions/123', {}, expect_errors=True).status_int == 404

    def test_valid(self):
        """
        Test that permission actually gets deleted
        """
        self.auth('basic1', 'basic1')

        assert Permission.query.filter(Permission.id == 1).count() == 1
        self.delete('/permissions/1', {}, expect_errors=True)
        assert Permission.query.filter(Permission.id == 1).count() == 0


class TestUpdatePermissionRoute(PermissionRouteTest):
    def setUp(self):
        super(TestUpdatePermissionRoute, self).setUp()

    def test_not_authenticated(self):
        assert self.put('/permissions/1', {}, expect_errors=True).status_int == 401

    def test_no_permission(self):
        """
        Test that user without EditPermissions-permission cannot update permission
        """
        self.auth('basic2', 'basic2')

        assert self.put('/permissions/1',
                        {
                            'name': "test_perm",
                            'description': 'Test perm is test perm',
                            'serviceId': 0,
                            'isGlobal': True
                        },
                        expect_errors=True).status_int == 403

    def test_non_existent_permission(self):
        """
        Test that editing non existent permission yields correct error
        """
        self.auth('basic1', 'basic1')

        assert self.put('/permissions/123',
                        {
                            'name': "test_perm",
                            'description': 'Test perm is test perm',
                            'serviceId': 0,
                            'isGlobal': True
                        },
                        expect_errors=True).status_int == 404

    def test_valid(self):
        """
        Test that permission actually gets updated
        """
        self.auth('basic1', 'basic1')

        perm = Permission.get(1)
        assert perm.name == ViewPermissions.name

        self.put('/permissions/1',
                 {
                     'name': 'test_perm',
                     'description': 'Test perm is test perm',
                     'serviceId': 0,
                     'isGlobal': True
                 })

        assert perm.name == 'test_perm'

if __name__ == "__main__":
    unittest.main()
